﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using UnityEngine.EventSystems;
using UnityEngine.Tilemaps;
using Cinemachine;
using System;
using System.Linq;
using System.Threading.Tasks;

public class GameController : MonoBehaviour
{
    private static GameController _Instance;

    public static GameController Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<GameController>();
            }

            return _Instance;
        }
    }

    //Public editor variables
    public GameObject PlayerOneObject;
    public GameObject PlayerTwoObject;
    public TextMeshProUGUI RoleText;
    public GameObject Timer;
    public GameObject GameObjects;
    public GameObject ReadyScreen;
    public GameObject ResultScreen;
    public GameObject RoundResultScreen;
    public GameObject HintUI;
    public CinemachineVirtualCamera PlayerVirtualCamera;
    public CinemachineVirtualCamera SpawnVirtualCamera;
    public CinemachineVirtualCamera HiderVirtualCamera;
    public int OptionalHidingSpotCount;

    //Static variables to be edited by menu
    public static Constants.PlayerRole PlayerOneRole { get; set; }
    public static Constants.PlayerRole PlayerTwoRole { get; set; }
    public static Sprite PlayerOneSprite { get; set; }
    public static Sprite PlayerTwoSprite { get; set; }
    public static int NumberOfRounds { get; set; }
    public static Constants.Difficulties Difficulty { get; set; }

    //Public variables that are set privately
    public GameObject HiderPlayerObject { get; private set; }
    public GameObject SeekerPlayerObject { get; private set; }
    public bool IsTimerPaused { get; set; }
    public GamePhase CurrentGamePhase { get; private set; }
    public GameObject HidingObject { get; private set; }
    public int CurrentRound { get; private set; } = 1;
    public float RoundTime { get; private set; }
    public List<TimerInterval> TimerIntervals { get; private set; } = new List<TimerInterval>();

    //Private variables
    private int _PlayerOneScore;
    private int _PlayerTwoScore;
    private List<Tuple<Vector3, HidableMetadata>> _HidableTiles;
    private Constants.CharacterLegendType _LastLegendType = Constants.CharacterLegendType.None;

    void Start()
    {
        //We're starting a new game, initialize the scores, parse the tilemap and generate the spots for this playthrough
        _PlayerOneScore = 0;
        _PlayerTwoScore = 0;
        ParseTilemap();
        GenerateHidingSpots();

        if (PlayerOneSprite != null)
        {
            //Set the player one sprite
            PlayerOneObject.GetComponent<SpriteRenderer>().sprite = PlayerOneSprite;
            PlayerOneObject.GetComponent<Animator>().runtimeAnimatorController = PlayerOneSprite.name.Contains("One") ? Resources.Load("CharacterOneController") as RuntimeAnimatorController : Resources.Load("CharacterTwoController") as RuntimeAnimatorController;
 
            if (PlayerTwoSprite != null)
            {
                //Set the player two sprite
                PlayerTwoObject.GetComponent<SpriteRenderer>().sprite = PlayerTwoSprite;
                PlayerTwoObject.GetComponent<Animator>().runtimeAnimatorController = PlayerTwoSprite.name.Contains("One") ? Resources.Load("CharacterOneController") as RuntimeAnimatorController : Resources.Load("CharacterTwoController") as RuntimeAnimatorController;
            }
        }
        
        //Setup the general hider and seeker objects (so roles are handled as roles independent from players)
        if (PlayerOneRole == Constants.PlayerRole.Hider)
        {
            HiderPlayerObject = PlayerOneObject;
            HiderPlayerObject.GetComponent<SpriteRenderer>().sprite = PlayerOneSprite;
            SeekerPlayerObject = PlayerTwoObject;
            SeekerPlayerObject.GetComponent<SpriteRenderer>().sprite = PlayerTwoSprite;
        }

        else
        {
            HiderPlayerObject = PlayerTwoObject;
            HiderPlayerObject.GetComponent<SpriteRenderer>().sprite = PlayerTwoSprite;
            SeekerPlayerObject = PlayerOneObject;
            SeekerPlayerObject.GetComponent<SpriteRenderer>().sprite = PlayerOneSprite;
        }

        //Add the controllers, setup the camera
        HiderPlayerObject.AddComponent<HiderController>();
        SeekerPlayerObject.AddComponent<SeekerController>();
        HiderVirtualCamera.Follow = HiderPlayerObject.transform;
        SpawnVirtualCamera.gameObject.SetActive(true);

        //All setup, let's go to the begin phase
        ChangeGamePhase(new BeginPhase());
    }

    private void Awake()
    {
        //Intervals we should tick at
        TimerIntervals.Add(new TimerIntervalValue(60000));  //60s
        TimerIntervals.Add(new TimerIntervalValue(30000));  //30s
        TimerIntervals.Add(new TimerIntervalValue(15000));  //15s
        TimerIntervals.Add(new TimerIntervalRange(0, 10000));   //Every sec between 10 - 1

        //Set camera variables
        HiderVirtualCamera.m_Lens.OrthographicSize = 0.8f;
        PlayerVirtualCamera.GetCinemachineComponent<CinemachineFramingTransposer>().m_DeadZoneWidth = 0.0f;
        PlayerVirtualCamera.GetCinemachineComponent<CinemachineFramingTransposer>().m_DeadZoneHeight = 0.0f;

        InputManager.Instance.InputStateChanged += OnInputStateChanged;
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if(CurrentGamePhase.GetType() == typeof(HidingPhase))
        {
            //Hiding, just update the legend to display what was last visible for the new input type
            SetCharacterLegend(e.NewState, _LastLegendType, Constants.PlayerRole.Hider);
        }

        else if (CurrentGamePhase.GetType() == typeof(SeekingPhase))
        {
            //Hiding, update the legend to display what was last visible and update the hint legend for the new input type
            SetCharacterLegend(e.NewState, _LastLegendType, Constants.PlayerRole.Seeker);
            SetHintLegend(e.NewState);
        }
    }
    
    void Update()
    {
        CurrentGamePhase.Update();
    }

    /// <summary>
    /// Changes from the current game phase to the new game phase, internally handling calls to OnEnd and OnStart
    /// </summary>
    /// <param name="newPhase">The new phase to become current</param>
    public void ChangeGamePhase(GamePhase newPhase)
    {
        if (CurrentGamePhase != null)
        {
            //We've got a current phase, end it
            CurrentGamePhase.OnEndCurrentPhase();
        }

        //Start the new phase and set it to be current
        newPhase.OnStartCurrentPhase();
        CurrentGamePhase = newPhase;
    }

    /// <summary>
    /// Sets the legend above the player's head as given and as appropriate for the input type
    /// </summary>
    /// <param name="inputType">The type of input determines what legend to show - PC controls or Xbox controls</param>
    /// <param name="legendType">The type legend info to show</param>
    /// <param name="player">The player who's legend we want to set</param>
    public void SetCharacterLegend(Constants.InputState inputType, Constants.CharacterLegendType legendType, Constants.PlayerRole player)
    {
        _LastLegendType = legendType;

        if (inputType == Constants.InputState.Controller)
        {
            //Moving to controller, hide PC
            HiderPlayerObject.FindChild("PCLegend").SetActive(false);
            SeekerPlayerObject.FindChild("PCLegend").SetActive(false);

            if (player == Constants.PlayerRole.Hider)
            {
                //Show only the hider's legend
                SeekerPlayerObject.FindChild("ControllerLegend").SetActive(false);
                HiderPlayerObject.FindChild("ControllerLegend").SetActive(true);

                if (legendType == Constants.CharacterLegendType.Hide)
                {
                    //Show only the hide legend
                    HiderPlayerObject.FindChild("ControllerLegend").FindChild("UnhideLegend").SetActive(false);
                    HiderPlayerObject.FindChild("ControllerLegend").FindChild("HideLegend").SetActive(true);
                }

                else if(legendType == Constants.CharacterLegendType.Unhide)
                {
                    //Show only the unhide legend
                    HiderPlayerObject.FindChild("ControllerLegend").FindChild("HideLegend").SetActive(false);
                    HiderPlayerObject.FindChild("ControllerLegend").FindChild("UnhideLegend").SetActive(true);
                }

                else
                {
                    //Don't show any legend
                    HiderPlayerObject.FindChild("ControllerLegend").FindChild("HideLegend").SetActive(false);
                    HiderPlayerObject.FindChild("ControllerLegend").FindChild("UnhideLegend").SetActive(false);
                }
            }

            else
            {
                //Show only the seeker's legend
                HiderPlayerObject.FindChild("ControllerLegend").SetActive(false);
                SeekerPlayerObject.FindChild("ControllerLegend").SetActive(true);

                if (legendType == Constants.CharacterLegendType.Hide)
                {
                    //Show only the hide legend
                    SeekerPlayerObject.FindChild("ControllerLegend").FindChild("UnhideLegend").SetActive(false);
                    SeekerPlayerObject.FindChild("ControllerLegend").FindChild("HideLegend").SetActive(true);
                }

                else if(legendType == Constants.CharacterLegendType.Unhide)
                {
                    //Show only the unhide legend
                    SeekerPlayerObject.FindChild("ControllerLegend").FindChild("HideLegend").SetActive(false);
                    SeekerPlayerObject.FindChild("ControllerLegend").FindChild("UnhideLegend").SetActive(true);
                }

                else
                {
                    //Don't show any legend
                    SeekerPlayerObject.FindChild("ControllerLegend").FindChild("HideLegend").SetActive(false);
                    SeekerPlayerObject.FindChild("ControllerLegend").FindChild("UnhideLegend").SetActive(false);
                }
            }
        }

        else
        {
            //Moving to PC, hide controller
            HiderPlayerObject.FindChild("ControllerLegend").SetActive(false);
            SeekerPlayerObject.FindChild("ControllerLegend").SetActive(false);

            if (player == Constants.PlayerRole.Hider)
            {
                //Show only the hider's legend
                SeekerPlayerObject.FindChild("PCLegend").SetActive(false);
                HiderPlayerObject.FindChild("PCLegend").SetActive(true);

                if (legendType == Constants.CharacterLegendType.Hide)
                {
                    //Show only the hide legend
                    HiderPlayerObject.FindChild("PCLegend").FindChild("UnhideLegend").SetActive(false);
                    HiderPlayerObject.FindChild("PCLegend").FindChild("HideLegend").SetActive(true);
                }

                else if (legendType == Constants.CharacterLegendType.Unhide)
                {
                    //Show only the unhide legend
                    HiderPlayerObject.FindChild("PCLegend").FindChild("HideLegend").SetActive(false);
                    HiderPlayerObject.FindChild("PCLegend").FindChild("UnhideLegend").SetActive(true);
                }

                else
                {
                    //Don't show any legend
                    HiderPlayerObject.FindChild("PCLegend").FindChild("HideLegend").SetActive(false);
                    HiderPlayerObject.FindChild("PCLegend").FindChild("UnhideLegend").SetActive(false);
                }
            }

            else
            {
                //Show only the seeker's legend
                HiderPlayerObject.FindChild("PCLegend").SetActive(false);
                SeekerPlayerObject.FindChild("PCLegend").SetActive(true);

                if (legendType == Constants.CharacterLegendType.Hide)
                {
                    //Show only the hide legend
                    SeekerPlayerObject.FindChild("PCLegend").FindChild("UnhideLegend").SetActive(false);
                    SeekerPlayerObject.FindChild("PCLegend").FindChild("HideLegend").SetActive(true);
                }

                else if (legendType == Constants.CharacterLegendType.Unhide)
                {
                    //Show only the unhide legend
                    SeekerPlayerObject.FindChild("PCLegend").FindChild("HideLegend").SetActive(false);
                    SeekerPlayerObject.FindChild("PCLegend").FindChild("UnhideLegend").SetActive(true);
                }

                else
                {
                    //Don't show any legend
                    SeekerPlayerObject.FindChild("PCLegend").FindChild("HideLegend").SetActive(false);
                    SeekerPlayerObject.FindChild("PCLegend").FindChild("UnhideLegend").SetActive(false);
                }
            }
        }
    }

    /// <summary>
    /// Sets the legend displayed on the hint panel
    /// </summary>
    /// <param name="inputType">The type of input determines what legend to show - PC controls or Xbox controls</param>
    public void SetHintLegend(Constants.InputState inputType)
    {
        if(inputType == Constants.InputState.Controller)
        {
            //Hide the PC legend, show the controller legend
            HintUI.FindChild("HintPanel").FindChild("PCLegend").SetActive(false);
            HintUI.FindChild("HintPanel").FindChild("ControllerLegend").SetActive(true);
        }

        else
        {
            //Hide the controller legend, show the PC legend
            HintUI.FindChild("HintPanel").FindChild("ControllerLegend").SetActive(false);
            HintUI.FindChild("HintPanel").FindChild("PCLegend").SetActive(true);
        }
    }

    /// <summary>
    /// Sets the hiding location to the hidingObject and begins the hide process
    /// </summary>
    /// <param name="hidingObject"></param>
    public void SetHideLocation(GameObject hidingObject)
    {
        //Hiding, so disable pausing and player movement
        PauseMenu.CanPause = false;
        HiderPlayerObject.GetComponent<BasicMovement>().CanMove = false;

        //Set the hiding object, begin animating the object
        HidingObject = hidingObject;
        HidingObject.GetComponent<HidableController>().Hide();

        //Finally, set the legend to display unhide info and begin fading the hider into the spot
        SetCharacterLegend(InputManager.Instance.InputState, Constants.CharacterLegendType.Unhide, Constants.PlayerRole.Hider);
        HiderPlayerObject.GetComponent<Animator>().Play("Fade");
    }

    /// <summary>
    /// Unsets the hiding location and cancels the hide process
    /// </summary>
    public void UnsetHideLocation()
    {
        //We're unhiding, so we can pause and move again - go back to idle, display the hide legend, animate the object and null the hiding object
        PauseMenu.CanPause = true;
        HiderPlayerObject.GetComponent<Animator>().Play("Idle");
        SetCharacterLegend(InputManager.Instance.InputState, Constants.CharacterLegendType.Hide, Constants.PlayerRole.Hider);
        HiderPlayerObject.GetComponent<BasicMovement>().CanMove = true;
        HidingObject.GetComponent<HidableController>().Unhide();
        HidingObject = null;
    }

    /// <summary>
    /// Confirms the hiding location for the hider and moves to the next game stage
    /// </summary>
    /// <param name="obj">An optional game object to force set as the hiding object</param>
    public async void ConfirmHideLocation(GameObject obj = null)
    {
        //We're confirmed, so disable the timer
        IsTimerPaused = true;
        Timer.GetComponent<Animator>().enabled = false;

        if (obj != null)
        {
            //Incase the hiding object is being force set
            HidingObject = obj;
        }
        
        //Cleanup the hider, go back to idle, remove the legend and set inactive
        HiderPlayerObject.GetComponent<Animator>().Play("Idle");
        SetCharacterLegend(InputManager.Instance.InputState, Constants.CharacterLegendType.None, Constants.PlayerRole.Hider);
        HiderPlayerObject.SetActive(false);

        //Wait for the object to be confirmed
        HidableController controller = HidingObject.GetComponent<HidableController>();
        await controller.Confirm();
        
        //Reset the tick animation
        Timer.GetComponent<Animator>().enabled = true;
        Timer.GetComponent<Animator>().Play("TimerTick", 0, 0);
        Timer.GetComponent<Animator>().enabled = false;

        //Time for the seeker to begin, enable them but disable the movement and update the role text
        SeekerPlayerObject.SetActive(true);
        SeekerPlayerObject.GetComponent<BasicMovement>().CanMove = false;
        RoleText.text = "SEEKER";

        //Set the hider zoom camera to the position of the hidable, don't follow the hider anymore
        HiderVirtualCamera.transform.position = new Vector3(HidingObject.transform.position.x, HidingObject.transform.position.y, -10.0f);
        PlayerVirtualCamera.Follow = null;

        //Now time to go into intermission
        ChangeGamePhase(new IntermissionPhase());
    }

    /// <summary>
    /// Returns a boolean representing whether or not the hider is in the collisionObject
    /// </summary>
    /// <param name="collisionObject">The object to check if the hider is in</param>
    /// <returns></returns>
    public bool TryUncoverHider(GameObject collisionObject)
    {
        return collisionObject == HidingObject;
    }

    /// <summary>
    /// Reveals the hider from their spot
    /// </summary>
    public void RevealHider()
    {
        //Setup the UI, disable the timer and the hider from moving
        ((SeekingPhase)CurrentGamePhase).CanGetHint = false;
        HintUI.SetActive(false);
        HiderPlayerObject.GetComponent<BasicMovement>().CanMove = false;
        SeekerPlayerObject.GetComponent<BasicMovement>().CanMove = false;
        IsTimerPaused = true;
        Timer.GetComponent<Animator>().enabled = false;

        //Reveal the hider
        HiderPlayerObject.SetActive(true);
        HiderPlayerObject.GetComponent<Animator>().Play("Reveal", -1, 0.0f);

        if (AudioManager.Instance != null && AudioManager.Instance.HiderRevealAudio != null)
        {
            AudioManager.Instance.PlaySFX(AudioManager.Instance.HiderRevealAudio);
        }

        //Do this just in case the Hider gets moved due to colliding with something when setting active again
        HiderVirtualCamera.Follow = null;
        HiderVirtualCamera.transform.position = HiderPlayerObject.transform.position;
        HiderVirtualCamera.Follow = HiderPlayerObject.transform;

        //Revealing, so blend to the hider camera
        PlayerVirtualCamera.BlendTo(HiderVirtualCamera);
    }

    /// <summary>
    /// Sets the round time for the specified role as per the set difficulty
    /// </summary>
    /// <param name="role">The role to set time for</param>
    public void SetRoundTime(Constants.PlayerRole role)
    {
        if (Difficulty == Constants.Difficulties.Easy)
        {
            if (role == Constants.PlayerRole.Hider)
            {
                RoundTime = GameManager.Instance.EasyHiderTime;
            }

            else
            {
                RoundTime = GameManager.Instance.EasySeekerTime;
            }
        }

        else if (Difficulty == Constants.Difficulties.Medium)
        {
            if (role == Constants.PlayerRole.Hider)
            {
                RoundTime = GameManager.Instance.MediumHiderTime;
            }

            else
            {
                RoundTime = GameManager.Instance.MediumSeekerTime;
            }
        }

        else
        {
            if (role == Constants.PlayerRole.Hider)
            {
                RoundTime = GameManager.Instance.HardHiderTime;
            }

            else
            {
                RoundTime = GameManager.Instance.HardSeekerTime;
            }
        }
    }

    /// <summary>
    /// Ends the current round, updating the score as appropriate and then moves onto the next appropriate game phase
    /// </summary>
    /// <param name="winnerOfRound">The role that won the round</param>
    public void EndCurrentRound(Constants.PlayerRole winnerOfRound)
    {
        HidingObject = null;

        //Update the score for the corresponding player
        if (winnerOfRound == Constants.PlayerRole.Hider)
        {
            if (PlayerOneRole == Constants.PlayerRole.Hider)
            {
                _PlayerOneScore++;
            }

            else
            {
                _PlayerTwoScore++;
            }
        }

        else
        {
            if (PlayerOneRole == Constants.PlayerRole.Seeker)
            {
                _PlayerOneScore++;
            }

            else
            {
                _PlayerTwoScore++;
            }
        }

        if (CurrentRound < NumberOfRounds)
        {
            //We've still got rounds left to play, so we're going to the end round phase
            EndRoundPhase endRoundPhase = new EndRoundPhase(winnerOfRound, CurrentRound);
            CurrentRound++;
            
            //Remove the old controllers, add the new ones and update the roles and objects
            if (PlayerOneRole == Constants.PlayerRole.Hider)
            {
                Destroy(PlayerOneObject.GetComponent<HiderController>());
                Destroy(PlayerTwoObject.GetComponent<SeekerController>());

                PlayerOneObject.AddComponent<SeekerController>();
                PlayerTwoObject.AddComponent<HiderController>();

                PlayerOneRole = Constants.PlayerRole.Seeker;
                PlayerTwoRole = Constants.PlayerRole.Hider;
                HiderPlayerObject = PlayerTwoObject;
                SeekerPlayerObject = PlayerOneObject;
            }

            else
            {
                Destroy(PlayerOneObject.GetComponent<SeekerController>());
                Destroy(PlayerTwoObject.GetComponent<HiderController>());

                PlayerOneObject.AddComponent<HiderController>();
                PlayerTwoObject.AddComponent<SeekerController>();

                PlayerOneRole = Constants.PlayerRole.Hider;
                PlayerTwoRole = Constants.PlayerRole.Seeker;
                HiderPlayerObject = PlayerOneObject;
                SeekerPlayerObject = PlayerTwoObject;
            }

            ChangeGamePhase(endRoundPhase);
        }

        else
        {
            //We've got no rounds left to play, it's game over
            GameOverPhase gameOverPhase;

            if (_PlayerOneScore > _PlayerTwoScore)
            {
                //Player One wins
                gameOverPhase = new GameOverPhase(1);
            }

            else
            {
                //Player Two wins
                gameOverPhase = new GameOverPhase(2);
            }

            ChangeGamePhase(gameOverPhase);
        }
    }

    /// <summary>
    /// Called when the game wants to be restarted
    /// </summary>
    public void PlayAgain()
    {
        HidingObject = null;
        CurrentRound = 1;

        //Remove the old controllers, add the new ones
        if (PlayerOneRole == Constants.PlayerRole.Seeker)
        {
            Destroy(PlayerOneObject.GetComponent<HiderController>());
            Destroy(PlayerTwoObject.GetComponent<SeekerController>());

            PlayerOneObject.AddComponent<SeekerController>();
            PlayerTwoObject.AddComponent<HiderController>();
        }

        else
        {
            Destroy(PlayerOneObject.GetComponent<SeekerController>());
            Destroy(PlayerTwoObject.GetComponent<HiderController>());

            PlayerOneObject.AddComponent<HiderController>();
            PlayerTwoObject.AddComponent<SeekerController>();
        }

        //Reset the score
        _PlayerOneScore = 0;
        _PlayerTwoScore = 0;
        ParseTilemap();

        //Update the objects
        if (PlayerOneRole == Constants.PlayerRole.Hider)
        {
            HiderPlayerObject = PlayerOneObject;
            SeekerPlayerObject = PlayerTwoObject;
        }

        else
        {
            HiderPlayerObject = PlayerTwoObject;
            SeekerPlayerObject = PlayerOneObject;
        }

        HiderPlayerObject.AddComponent<HiderController>();
        SeekerPlayerObject.AddComponent<SeekerController>();
        ChangeGamePhase(new BeginPhase());
    }

    /// <summary>
    /// Parses the tilemap to find and replace all hidable locations on the hidables tilemap
    /// </summary>
    private void ParseTilemap()
    {
        _HidableTiles = new List<Tuple<Vector3, HidableMetadata>>();
        Tilemap hidablesTilemap = GameObject.Find("hidablesTilemap").GetComponent<Tilemap>();

        //We'll replace with a blank tile
        Tile replacementTile = (Tile)ScriptableObject.CreateInstance(typeof(Tile));
        replacementTile.sprite = Resources.Load("blank") as Sprite;
        List<TileBase> tilesToSwap = new List<TileBase>();

        for (int n = hidablesTilemap.cellBounds.xMin; n < hidablesTilemap.cellBounds.xMax; n++)
        {
            for (int p = hidablesTilemap.cellBounds.yMin; p < hidablesTilemap.cellBounds.yMax; p++)
            {
                Vector3Int localPlace = (new Vector3Int(n, p, (int)hidablesTilemap.transform.position.y));
                Vector3 place = hidablesTilemap.CellToWorld(localPlace) + new Vector3(0.0f, 0.365f, 0.0f);

                if (hidablesTilemap.HasTile(localPlace))
                {
                    //We've got a tile in this location
                    TileBase tile = hidablesTilemap.GetTile(localPlace);

                    if (HidablesManager.Instance.HidableGameObjects.ContainsKey(tile.name))
                    {
                        //It's a hidable tile, let's get its metadata, add it to the hidable tiles and the tiles to swap with blank
                        HidableMetadata thisHidableMetadata = HidablesManager.Instance.HidableGameObjects[tile.name];
                        _HidableTiles.Add(new Tuple<Vector3, HidableMetadata>(place, thisHidableMetadata));
                        tilesToSwap.Add(tile);
                    }
                }
            }
        }

        foreach (TileBase tileToSwap in tilesToSwap)
        {
            //Got all our hidables now, let's replace all the tiles that have the text on them with blank
            hidablesTilemap.SwapTile(tileToSwap, replacementTile);
        }
    }

    /// <summary>
    /// Loops through the hidable tiles, spawning the hidable spots into the game depending on whether they are optional or not
    /// </summary>
    private void GenerateHidingSpots()
    {
        List<Tuple<Vector3, HidableMetadata>> optionalTiles = new List<Tuple<Vector3, HidableMetadata>>();

        foreach (Tuple<Vector3, HidableMetadata> tile in _HidableTiles)
        {
            if (!tile.Item2.Controller.IsOptionalHidable)
            {
                //It's not optional, spawn it
                GameObject hidableObject = GameObject.Instantiate(tile.Item2.GameObject);
                hidableObject.transform.position = tile.Item1 + hidableObject.transform.position;
                hidableObject.transform.parent = GameObjects.transform;
            }

            else
            {
                //It's optional, add it to the list
                optionalTiles.Add(tile);
            }
        }

        System.Random rand = new System.Random(Guid.NewGuid().GetHashCode());
        foreach (Tuple<Vector3, HidableMetadata> tile in optionalTiles.OrderBy(x => rand.Next()).Take(OptionalHidingSpotCount))
        {
            //Grab a random selection of hidables from the list based on the count, and spawn them all in
            GameObject hidableObject = GameObject.Instantiate(tile.Item2.GameObject);
            hidableObject.transform.position = tile.Item1 + hidableObject.transform.position;
            hidableObject.transform.parent = GameObjects.transform;
        }
    }

    /// <summary>
    /// Displays a hint to the seeker
    /// </summary>
    public void DisplayHint()
    {
        //Show the hint UI
        HintUI.FindChild("HintPanel").SetActive(false);
        HintUI.FindChild("HintDisplay").SetActive(true);

        //Calculate the distance and the amount to fill the hint image
        float dist = Mathf.Abs(Vector3.Distance(HidingObject.transform.position, SeekerPlayerObject.transform.position));

        float fillAmount = ((500 - (dist * 50.0f)) / 500.0f);

        //Clamping
        if (fillAmount > 1.0f)
        {
            fillAmount = 1.0f;
        }

        else if (fillAmount < 0.05f)
        {
            fillAmount = 0.05f;
        }
        
        if(fillAmount >= 0.5f)
        {
            //We're hot, play the hot SFX and show the hot sprites
            if(AudioManager.Instance != null && AudioManager.Instance.HintHotAudio != null)
            {
                AudioManager.Instance.PlaySFX(AudioManager.Instance.HintHotAudio);
            }

            if(HiderPlayerObject.GetComponent<SpriteRenderer>().sprite.name.Contains("One"))
            {
                HintUI.FindChild("HintDisplay").GetComponent<Animator>().Play("CharacterOneHot");
            }

            else
            {
                HintUI.FindChild("HintDisplay").GetComponent<Animator>().Play("CharacterTwoHot");
            }
        }

        else
        {
            //We're cold, play the cold SFX and show the cold sprites
            if (AudioManager.Instance != null && AudioManager.Instance.HintColdAudio != null)
            {
                AudioManager.Instance.PlaySFX(AudioManager.Instance.HintColdAudio);
            }

            if (HiderPlayerObject.GetComponent<SpriteRenderer>().sprite.name.Contains("One"))
            {
                HintUI.FindChild("HintDisplay").GetComponent<Animator>().Play("CharacterOneCold");
            }

            else
            {
                HintUI.FindChild("HintDisplay").GetComponent<Animator>().Play("CharacterTwoCold");
            }
        }

        //Set the amount of the hint image to display and set the correct colour
        Image image = HintUI.FindChild("HintDisplay").FindChild("Fill").GetComponent<Image>();
        image.fillAmount = fillAmount;
        image.color = Color.Lerp(Color.blue, Color.red, fillAmount);
    }
}
